# Complete Deployment Guide
## Multi-Tool Website with AI Presentation & Assignment Generators

---

## 📦 Package Contents

### Backend (`/backend`)
- `app_complete.py` - Complete Flask API server
- `requirements_complete.txt` - All Python dependencies
- `/tools/` - Tool implementation modules:
  - `presentation_generator_v2.py` - AI Presentation Generator
  - `assignment_generator_v2.py` - AI Assignment Generator
  - `logo_analyzer.py` - Logo Analysis
  - `screenshot_ocr.py` - OCR Text Extraction
  - `resume_bullet_generator.py` - Resume Bullets
  - `pdf_cleaner.py` - PDF Page Cleaner
  - `caption_rewriter.py` - Caption Rewriter

### Frontend (`/frontend`)
- `/pages/` - All HTML pages including:
  - `ai-presentation-generator.html`
  - `ai-assignment-generator.html`
  - All existing tool pages
- `/css/style.css` - Unified stylesheet
- `/js/main.js` - Global JavaScript
- `index.html` - Homepage

---

## 🚀 Quick Start Deployment

### Option 1: Local Development

#### Step 1: Install Python Dependencies
```bash
cd backend
python3 -m pip install -r requirements_complete.txt
```

#### Step 2: Install System Dependencies (for OCR)
**Ubuntu/Debian:**
```bash
sudo apt-get update
sudo apt-get install -y tesseract-ocr tesseract-ocr-eng
```

**macOS:**
```bash
brew install tesseract
```

**Windows:**
- Download from: https://github.com/UB-Mannheim/tesseract/wiki
- Add to PATH

#### Step 3: Set Environment Variables
Create `.env` file in `/backend`:
```bash
# Required for AI tools
GEMINI_API_KEY=your_gemini_api_key_here

# Optional - for better images (free tiers available)
UNSPLASH_ACCESS_KEY=your_unsplash_key
PEXELS_API_KEY=your_pexels_key
```

**Get Free API Keys:**
- **Gemini AI** (Required): https://makersuite.google.com/app/apikey
- **Unsplash** (Optional): https://unsplash.com/developers
- **Pexels** (Optional): https://www.pexels.com/api/

#### Step 4: Start Backend Server
```bash
cd backend
python3 app_complete.py
```
Server runs on: `http://localhost:5000`

#### Step 5: Start Frontend
```bash
cd frontend
# Option A: Python simple server
python3 -m http.server 8000

# Option B: Node.js
npx serve .

# Option C: PHP
php -S localhost:8000
```
Frontend runs on: `http://localhost:8000`

---

### Option 2: Production Deployment

#### Backend Deployment (Heroku Example)

1. **Prepare Heroku Files**

`Procfile`:
```
web: gunicorn app_complete:app
```

`runtime.txt`:
```
python-3.11.5
```

Add to `requirements_complete.txt`:
```
gunicorn==21.2.0
```

2. **Deploy to Heroku**
```bash
cd backend
heroku create your-app-name
heroku config:set GEMINI_API_KEY=your_key_here
git push heroku main
```

#### Frontend Deployment Options

**Option A: Netlify**
1. Connect GitHub repository
2. Build settings: None (static site)
3. Publish directory: `frontend`
4. Update API_BASE_URL in HTML files to your Heroku backend URL

**Option B: Vercel**
```bash
cd frontend
vercel deploy
```

**Option C: GitHub Pages**
1. Push `frontend` folder to repository
2. Enable GitHub Pages in settings
3. Update API URLs

#### Update API URLs in Frontend
In all HTML files, change:
```javascript
const API_BASE_URL = 'http://localhost:5000/api';
```
To:
```javascript
const API_BASE_URL = 'https://your-backend-app.herokuapp.com/api';
```

---

## 🔧 Configuration Guide

### Backend Configuration

#### Environment Variables
```bash
# .env file
GEMINI_API_KEY=AIzaSy...              # Required for AI generation
UNSPLASH_ACCESS_KEY=xyz123...         # Optional for presentation images
PEXELS_API_KEY=abc456...              # Optional backup for images
FLASK_ENV=production                   # Set to production for deployment
MAX_CONTENT_LENGTH=52428800           # Max upload size (50MB)
```

#### CORS Configuration
If frontend and backend are on different domains, ensure CORS is properly configured in `app_complete.py`:

```python
from flask_cors import CORS

# Allow specific origins
CORS(app, origins=["https://your-frontend-domain.com"])
```

### Frontend Configuration

#### API Endpoint Configuration
Update in each tool HTML file:
```javascript
// Development
const API_BASE_URL = 'http://localhost:5000/api';

// Production
const API_BASE_URL = 'https://api.yourdomain.com/api';
```

---

## 📋 QA Testing Checklist

### ✅ AI Presentation Generator
- [ ] Topic input validation (minimum 3 characters)
- [ ] Slide count validation (3-20 range)
- [ ] Outline generation works with AI
- [ ] Outline displays correctly with all fields
- [ ] PPTX format generates and downloads
- [ ] PDF format generates and downloads
- [ ] DOCX format generates and downloads
- [ ] Watermark toggle works correctly
- [ ] Images fetch from Unsplash/Pexels (if API keys provided)
- [ ] Placeholder images work without API keys
- [ ] Speaker notes included in outputs
- [ ] File downloads with correct filename
- [ ] Loading indicators display during generation
- [ ] Error messages display for failures
- [ ] Mobile responsive design
- [ ] Dark/light mode compatibility

### ✅ AI Assignment Generator
- [ ] Topic input validation (minimum 5 characters)
- [ ] Page count validation (5-30 range)
- [ ] Word count calculator updates correctly
- [ ] TOC generation works with AI
- [ ] TOC displays with proper structure
- [ ] DOCX format generates complete document
- [ ] PDF format generates complete document
- [ ] Title page included
- [ ] Table of contents formatted correctly
- [ ] Introduction generated
- [ ] Multiple chapters generated based on page count
- [ ] Conclusion included
- [ ] APA references formatted correctly
- [ ] Word count approximately 300-350 per page
- [ ] Content is unique and contextual
- [ ] File downloads with correct filename
- [ ] Loading indicator works (1-3 min generation)
- [ ] Mobile responsive design
- [ ] Dark/light mode compatibility

### ✅ Existing Tools (Fixed)
- [ ] Logo Analyzer: Uploads and analyzes images
- [ ] Screenshot OCR: Extracts text accurately
- [ ] Resume Bullets: Generates professional bullets
- [ ] Image DPI Converter: Converts and updates metadata
- [ ] PDF Page Cleaner: Extracts specific pages
- [ ] Caption Rewriter: Rewrites for different platforms
- [ ] All tools have Clear/Reset buttons
- [ ] No default placeholder text
- [ ] Proper validation messages
- [ ] Download functionality works
- [ ] Mobile friendly UI
- [ ] Clean error handling

### ✅ Global Requirements
- [ ] No default text in input fields
- [ ] Clear/Reset buttons on all tools
- [ ] Mobile-friendly responsive design
- [ ] Form validation with clear error messages
- [ ] Reliable file downloads
- [ ] Fast loading times
- [ ] No broken links
- [ ] Dark/light mode persistent
- [ ] Navigation works on all pages
- [ ] Footer displays correctly
- [ ] Contact form UI works
- [ ] Cross-browser compatibility (Chrome, Firefox, Safari, Edge)

---

## 🔍 Testing Commands

### Test Backend API
```bash
# Health check
curl http://localhost:5000/health

# Test presentation outline
curl -X POST http://localhost:5000/api/presentation/generate-outline \
  -H "Content-Type: application/json" \
  -d '{"topic": "AI in Healthcare", "slide_count": 5}'

# Test assignment TOC
curl -X POST http://localhost:5000/api/assignment/generate-toc \
  -H "Content-Type: application/json" \
  -d '{"topic": "Climate Change", "page_count": 10}'
```

### Test Frontend
1. Open browser dev tools (F12)
2. Check Console for errors
3. Test each tool manually
4. Verify API calls in Network tab
5. Test on mobile device/emulator

---

## 🐛 Troubleshooting

### Common Issues

#### Backend Won't Start
**Error:** `Module not found`
- **Solution:** Install dependencies: `pip install -r requirements_complete.txt`

**Error:** `Port already in use`
- **Solution:** Kill process on port 5000: `lsof -ti:5000 | xargs kill -9`

#### AI Generation Fails
**Error:** `API key not configured`
- **Solution:** Add GEMINI_API_KEY to `.env` file or environment variables

**Error:** `Content generation timeout`
- **Solution:** Check internet connection, verify API quota not exceeded

#### File Downloads Fail
**Error:** `File not found`
- **Solution:** Check `/tmp/presentations/` and `/tmp/assignments/` directories exist
- **Solution:** Verify write permissions on temp directories

#### CORS Errors
**Error:** `Access-Control-Allow-Origin`
- **Solution:** Update CORS configuration in `app_complete.py`
- **Solution:** Ensure backend URL matches in frontend code

#### OCR Not Working
**Error:** `Tesseract not found`
- **Solution:** Install Tesseract OCR system package
- **Solution:** Add Tesseract to system PATH

---

## 📊 Performance Optimization

### Backend
- Use gunicorn with multiple workers: `gunicorn -w 4 app_complete:app`
- Enable caching for static content
- Compress responses with gzip
- Use CDN for static assets

### Frontend
- Minimize CSS/JS files
- Compress images
- Use lazy loading
- Enable browser caching
- Use CDN for fonts

---

## 🔒 Security Considerations

1. **API Keys**: Never commit `.env` file to version control
2. **File Uploads**: Validate file types and sizes
3. **CORS**: Restrict to specific domains in production
4. **Rate Limiting**: Implement rate limiting on API endpoints
5. **HTTPS**: Always use HTTPS in production
6. **Temp Files**: Clean up old generated files periodically

---

## 📞 Support & Resources

### API Documentation
- **Gemini AI**: https://ai.google.dev/docs
- **Unsplash API**: https://unsplash.com/documentation
- **Pexels API**: https://www.pexels.com/api/documentation

### Libraries Used
- **Flask**: https://flask.palletsprojects.com/
- **python-pptx**: https://python-pptx.readthedocs.io/
- **python-docx**: https://python-docx.readthedocs.io/
- **ReportLab**: https://www.reportlab.com/docs/reportlab-userguide.pdf

### Getting Help
1. Check console logs for errors
2. Verify all dependencies installed
3. Test API endpoints individually
4. Review environment variables
5. Check file permissions

---

## 📝 Maintenance

### Regular Tasks
- Monitor API usage quotas
- Clean temporary files: `rm -rf /tmp/presentations/* /tmp/assignments/*`
- Update dependencies: `pip install --upgrade -r requirements_complete.txt`
- Review error logs
- Backup generated content if needed

### Scaling
- Add Redis for caching
- Use Celery for background tasks
- Implement queue for AI generation
- Add load balancer for multiple backend instances

---

**Last Updated:** 2024
**Version:** 2.0.0
**Status:** ✅ Production Ready
