"""
AI Presentation Generator (Gamma-style)
Generates professional presentations with auto-fetched images
Uses free Unsplash API for images
"""

from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.enum.text import PP_ALIGN
from pptx.dml.color import RGBColor
import requests
import os
from io import BytesIO
from PIL import Image
from reportlab.lib.pagesizes import letter
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Image as RLImage
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from docx import Document
from docx.shared import Inches as DocxInches

# Unsplash configuration
UNSPLASH_ACCESS_KEY = os.environ.get('UNSPLASH_ACCESS_KEY', 'YOUR_KEY_HERE')
UNSPLASH_API = 'https://api.unsplash.com/search/photos'

def generate_outline(topic, slide_count):
    """Generate presentation outline based on topic"""
    # Common presentation structure
    outlines = {
        10: [
            f"Introduction to {topic}",
            f"What is {topic}?",
            f"Key Concepts",
            f"Benefits and Advantages",
            f"Challenges and Solutions",
            f"Real-world Applications",
            f"Case Studies",
            f"Best Practices",
            f"Future Trends",
            f"Conclusion and Takeaways"
        ],
        15: [
            f"Introduction to {topic}",
            f"Background and Context",
            f"Core Principles",
            f"Key Components Part 1",
            f"Key Components Part 2",
            f"Benefits and Impact",
            f"Common Challenges",
            f"Solution Approaches",
            f"Implementation Strategy",
            f"Case Study 1",
            f"Case Study 2",
            f"Best Practices",
            f"Tools and Resources",
            f"Future Outlook",
            f"Summary and Next Steps"
        ],
        20: [
            f"Title: {topic}",
            f"Agenda",
            f"Introduction",
            f"Background",
            f"Problem Statement",
            f"Research Overview",
            f"Methodology",
            f"Key Findings Part 1",
            f"Key Findings Part 2",
            f"Key Findings Part 3",
            f"Analysis",
            f"Implications",
            f"Benefits",
            f"Challenges",
            f"Solutions",
            f"Case Studies",
            f"Recommendations",
            f"Implementation Plan",
            f"Conclusion",
            f"Q&A and Contact"
        ]
    }
    
    return outlines.get(slide_count, outlines[10][:slide_count])

def generate_slide_content(title, topic):
    """Generate bullet points for a slide"""
    # Extract key words from title
    keywords = title.lower().split()
    
    bullets = [
        f"Understanding {title.lower()} in the context of {topic}",
        f"Key factors that influence {title.lower()} outcomes",
        f"Practical applications and real-world examples",
        f"Best practices for implementing {title.lower()} strategies",
        f"Measuring success and continuous improvement"
    ]
    
    # Customize based on title keywords
    if 'introduction' in keywords or 'what' in keywords:
        bullets = [
            f"Definition and core concepts of {topic}",
            f"Historical context and evolution",
            f"Why {topic} matters today",
            f"Key stakeholders and applications",
            f"Overview of what we'll cover"
        ]
    
    elif 'benefit' in keywords or 'advantage' in keywords:
        bullets = [
            f"Increased efficiency and productivity",
            f"Cost reduction and resource optimization",
            f"Improved outcomes and quality",
            f"Competitive advantages in the market",
            f"Long-term sustainable impact"
        ]
    
    elif 'challenge' in keywords or 'problem' in keywords:
        bullets = [
            f"Common obstacles and barriers",
            f"Resource constraints and limitations",
            f"Technical and operational challenges",
            f"Adoption and change management issues",
            f"Risk factors to consider"
        ]
    
    elif 'solution' in keywords or 'approach' in keywords:
        bullets = [
            f"Proven strategies and methodologies",
            f"Step-by-step implementation framework",
            f"Tools and technologies to leverage",
            f"Team collaboration and communication",
            f"Monitoring and optimization tactics"
        ]
    
    elif 'future' in keywords or 'trend' in keywords:
        bullets = [
            f"Emerging technologies and innovations",
            f"Predicted market developments",
            f"Opportunities for growth and expansion",
            f"Preparing for upcoming changes",
            f"Long-term vision and roadmap"
        ]
    
    elif 'conclusion' in keywords or 'summary' in keywords:
        bullets = [
            f"Key takeaways from {topic}",
            f"Action items and next steps",
            f"Resources for further learning",
            f"Questions and discussion points",
            f"Thank you and contact information"
        ]
    
    return bullets[:5]  # Limit to 5 bullets

def generate_speaker_notes(title):
    """Generate speaker notes for a slide"""
    return f"When presenting this slide about '{title}', emphasize the key points and provide real-world examples where applicable. Encourage audience engagement through questions."

def fetch_image_from_unsplash(keyword, slide_number):
    """Fetch free image from Unsplash"""
    try:
        params = {
            'query': keyword,
            'client_id': UNSPLASH_ACCESS_KEY,
            'per_page': 1,
            'orientation': 'landscape'
        }
        
        response = requests.get(UNSPLASH_API, params=params, timeout=10)
        
        if response.status_code == 200:
            data = response.json()
            if data['results']:
                image_url = data['results'][0]['urls']['regular']
                
                # Download image
                img_response = requests.get(image_url, timeout=10)
                img = Image.open(BytesIO(img_response.content))
                
                # Save temporarily
                temp_path = f"outputs/slide_{slide_number}_image.jpg"
                os.makedirs('outputs', exist_ok=True)
                img.save(temp_path, 'JPEG')
                
                return temp_path
        
        # Fallback to placeholder
        return create_placeholder_image(slide_number)
        
    except Exception as e:
        print(f"Image fetch failed for '{keyword}': {e}")
        return create_placeholder_image(slide_number)

def create_placeholder_image(slide_number):
    """Create placeholder image if API fails"""
    from PIL import Image, ImageDraw, ImageFont
    
    img = Image.new('RGB', (1200, 675), color=(99, 102, 241))
    draw = ImageDraw.Draw(img)
    
    try:
        font = ImageFont.truetype("/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf", 80)
    except:
        font = ImageFont.load_default()
    
    text = f"Slide {slide_number}"
    bbox = draw.textbbox((0, 0), text, font=font)
    text_width = bbox[2] - bbox[0]
    text_height = bbox[3] - bbox[1]
    
    position = ((1200 - text_width) / 2, (675 - text_height) / 2)
    draw.text(position, text, fill=(255, 255, 255), font=font)
    
    path = f"outputs/placeholder_{slide_number}.jpg"
    os.makedirs('outputs', exist_ok=True)
    img.save(path, 'JPEG')
    
    return path

def create_pptx(slides_data, topic, add_watermark):
    """Generate PowerPoint presentation"""
    prs = Presentation()
    prs.slide_width = Inches(10)
    prs.slide_height = Inches(7.5)
    
    # Title slide
    title_slide_layout = prs.slide_layouts[0]
    slide = prs.slides.add_slide(title_slide_layout)
    title = slide.shapes.title
    subtitle = slide.placeholders[1]
    
    title.text = topic
    subtitle.text = "Professional Presentation"
    
    # Content slides
    for slide_data in slides_data:
        bullet_slide_layout = prs.slide_layouts[1]
        slide = prs.slides.add_slide(bullet_slide_layout)
        
        # Title
        title = slide.shapes.title
        title.text = slide_data['title']
        
        # Bullets
        content = slide.placeholders[1]
        text_frame = content.text_frame
        text_frame.clear()
        
        for bullet in slide_data['bullets']:
            p = text_frame.add_paragraph()
            p.text = bullet
            p.level = 0
            p.font.size = Pt(18)
        
        # Add image
        if slide_data.get('image_path') and os.path.exists(slide_data['image_path']):
            try:
                slide.shapes.add_picture(
                    slide_data['image_path'],
                    Inches(6.5), Inches(2),
                    width=Inches(3), height=Inches(4)
                )
            except:
                pass
        
        # Add watermark
        if add_watermark:
            try:
                text_box = slide.shapes.add_textbox(
                    Inches(8.5), Inches(7),
                    Inches(1.3), Inches(0.3)
                )
                text_frame = text_box.text_frame
                text_frame.text = "Multi-Tool"
                p = text_frame.paragraphs[0]
                p.font.size = Pt(10)
                p.font.color.rgb = RGBColor(150, 150, 150)
            except:
                pass
    
    # Save
    output_path = f"outputs/presentation_{topic[:30].replace(' ', '_')}.pptx"
    prs.save(output_path)
    
    return output_path

def create_pdf(slides_data, topic):
    """Generate PDF presentation"""
    output_path = f"outputs/presentation_{topic[:30].replace(' ', '_')}.pdf"
    
    doc = SimpleDocTemplate(output_path, pagesize=letter)
    styles = getSampleStyleSheet()
    story = []
    
    # Title
    title_style = ParagraphStyle(
        'CustomTitle',
        parent=styles['Heading1'],
        fontSize=24,
        textColor='#6366f1',
        spaceAfter=30,
        alignment=1  # Center
    )
    story.append(Paragraph(topic, title_style))
    story.append(Spacer(1, 0.5*inch))
    
    # Slides
    for slide_data in slides_data:
        # Slide title
        story.append(Paragraph(slide_data['title'], styles['Heading2']))
        story.append(Spacer(1, 0.2*inch))
        
        # Bullets
        for bullet in slide_data['bullets']:
            story.append(Paragraph(f"• {bullet}", styles['Normal']))
        
        story.append(Spacer(1, 0.3*inch))
        
        # Image
        if slide_data.get('image_path') and os.path.exists(slide_data['image_path']):
            try:
                story.append(RLImage(slide_data['image_path'], width=3*inch, height=2*inch))
            except:
                pass
        
        story.append(Spacer(1, 0.5*inch))
    
    doc.build(story)
    return output_path

def create_docx(slides_data, topic):
    """Generate Word document"""
    doc = Document()
    
    # Title
    doc.add_heading(topic, 0)
    doc.add_paragraph()
    
    # Slides
    for i, slide_data in enumerate(slides_data, 1):
        doc.add_heading(f"{i}. {slide_data['title']}", level=1)
        
        for bullet in slide_data['bullets']:
            doc.add_paragraph(bullet, style='List Bullet')
        
        # Image
        if slide_data.get('image_path') and os.path.exists(slide_data['image_path']):
            try:
                doc.add_picture(slide_data['image_path'], width=DocxInches(4))
            except:
                pass
        
        doc.add_paragraph()
    
    output_path = f"outputs/presentation_{topic[:30].replace(' ', '_')}.docx"
    doc.save(output_path)
    
    return output_path

def generate_presentation(topic, slide_count, format_type, add_watermark, outline=None):
    """
    Main presentation generation function
    
    Args:
        topic: Presentation topic
        slide_count: Number of slides (10/15/20)
        format_type: 'pptx', 'pdf', or 'docx'
        add_watermark: Boolean
        outline: Optional custom outline
    
    Returns:
        Path to generated file
    """
    try:
        # Generate or use provided outline
        if outline is None:
            outline = generate_outline(topic, slide_count)
        
        # Generate slide content
        slides_data = []
        for i, title in enumerate(outline, 1):
            bullets = generate_slide_content(title, topic)
            notes = generate_speaker_notes(title)
            
            # Fetch image
            image_keyword = f"{topic} {title}"
            image_path = fetch_image_from_unsplash(image_keyword, i)
            
            slides_data.append({
                'title': title,
                'bullets': bullets,
                'notes': notes,
                'image_path': image_path
            })
        
        # Generate file based on format
        if format_type == 'pptx':
            output_path = create_pptx(slides_data, topic, add_watermark)
        elif format_type == 'pdf':
            output_path = create_pdf(slides_data, topic)
        elif format_type == 'docx':
            output_path = create_docx(slides_data, topic)
        else:
            raise ValueError(f"Unsupported format: {format_type}")
        
        # Clean up temp images
        for slide_data in slides_data:
            if slide_data.get('image_path') and os.path.exists(slide_data['image_path']):
                try:
                    os.remove(slide_data['image_path'])
                except:
                    pass
        
        return output_path
        
    except Exception as e:
        raise Exception(f"Presentation generation failed: {str(e)}")

# For testing
if __name__ == "__main__":
    pass
